# frozen_string_literal: true

#
# Copyright (C) 2011 - present Instructure, Inc.
#
# This file is part of Canvas.
#
# Canvas is free software: you can redistribute it and/or modify it under
# the terms of the GNU Affero General Public License as published by the Free
# Software Foundation, version 3 of the License.
#
# Canvas is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
# details.
#
# You should have received a copy of the GNU Affero General Public License along
# with this program. If not, see <http://www.gnu.org/licenses/>.
#

require File.expand_path(File.dirname(__FILE__) + '/../spec_helper')

describe TokensController do
  describe "developer keys" do
    context "not logged in" do
      it "should require being logged in to create an access token" do
        post 'create', params: {:access_token => {:purpose => "test"}}
        expect(response).to be_redirect
        expect(assigns[:token]).to be_nil
      end

      it "should require being logged in to delete an access token" do
        delete 'destroy', params: {:id => 5}
        expect(response).to be_redirect
      end

      it "should require being logged in to retrieve an access token" do
        get 'show', params: {:id => 5}
        expect(response).to be_redirect
      end
    end

    context "logged in" do
      before :once do
        user_factory(active_user: true)
      end

      before :each do
        user_session(@user)
      end

      it "should allow creating an access token" do
        post 'create', params: {:access_token => {:purpose => "test", :expires_at => "jun 1 2011"}}
        expect(response).to be_successful
        expect(assigns[:token]).not_to be_nil
        expect(assigns[:token].developer_key).to eq DeveloperKey.default
        expect(assigns[:token].purpose).to eq "test"
        expect(assigns[:token].expires_at.to_date).to eq Time.parse("jun 1 2011").to_date
      end

      it "should not allow creating an access token while masquerading" do
        Account.site_admin.account_users.create!(user: @user)
        session[:become_user_id] = user_with_pseudonym.id

        post 'create', params: {:access_token => {:purpose => "test", :expires_at => "jun 1 2011"}}
        assert_status(401)
      end

      it "should not allow explicitly setting the token value" do
        post 'create', params: {:access_token => {:purpose => "test", :expires_at => "jun 1 2011", :token => "mytoken"}}
        expect(response).to be_successful
        expect(response.body).not_to match(/mytoken/)
        expect(assigns[:token]).not_to be_nil
        expect(assigns[:token].full_token).not_to match(/mytoken/)
        expect(response.body).to match(/#{assigns[:token].full_token}/)
        expect(assigns[:token].developer_key).to eq DeveloperKey.default
        expect(assigns[:token].purpose).to eq "test"
        expect(assigns[:token].expires_at.to_date).to eq Time.parse("jun 1 2011").to_date
      end

      it "should allow deleting an access token" do
        token = @user.access_tokens.create!
        expect(token.user_id).to eq @user.id
        delete 'destroy', params: {:id => token.id}
        expect(response).to be_successful
        expect(assigns[:token]).to be_deleted
      end

      it "should not allow deleting an access token while masquerading" do
        token = @user.access_tokens.create!
        expect(token.user_id).to eq @user.id
        Account.site_admin.account_users.create!(user: @user)
        session[:become_user_id] = user_with_pseudonym.id

        delete 'destroy', params: {:id => token.id}
        assert_status(401)
      end

      it "should not allow deleting someone else's access token" do
        user2 = User.create!
        token = user2.access_tokens.create!
        expect(token.user_id).to eq user2.id
        delete 'destroy', params: {:id => token.id}
        assert_status(404)
      end

      it "should allow retrieving an access token, but not give the full token string" do
        token = @user.access_tokens.new
        token.developer_key = DeveloperKey.default
        token.save!
        expect(token.user_id).to eq @user.id
        expect(token.protected_token?).to eq false
        get 'show', params: {:id => token.id}
        expect(response).to be_successful
        expect(assigns[:token]).to eq token
        expect(response.body).to match(/#{assigns[:token].token_hint}/)
      end

      it "should not include token for protected tokens" do
        key = DeveloperKey.create!
        token = @user.access_tokens.create!(developer_key: key)
        expect(token.user_id).to eq @user.id
        expect(token.protected_token?).to eq true
        get 'show', params: {:id => token.id}
        expect(response).to be_successful
        expect(assigns[:token]).to eq token
        expect(response.body).not_to match(/#{assigns[:token].token_hint}/)
      end

      it "should not allow retrieving someone else's access token" do
        user2 = User.create!
        token = user2.access_tokens.create!
        expect(token.user_id).to eq user2.id
        get 'show', params: {:id => token.id}
        assert_status(404)
      end

      it "should allow updating a token" do
        token = @user.access_tokens.new
        token.developer_key = DeveloperKey.default
        token.save!
        expect(token.user_id).to eq @user.id
        expect(token.protected_token?).to eq false
        put 'update', params: {:id => token.id, :access_token => {:purpose => 'new purpose'}}
        expect(response).to be_successful
        expect(assigns[:token]).to eq token
        expect(assigns[:token].purpose).to eq "new purpose"
        expect(response.body).to match(/#{assigns[:token].token_hint}/)
      end

      it "should allow regenerating an unprotected token" do
        token = @user.access_tokens.new
        token.developer_key = DeveloperKey.default
        token.save!
        expect(token.user_id).to eq @user.id
        expect(token.protected_token?).to eq false
        put 'update', params: {:id => token.id, :access_token => {:regenerate => '1'}}
        expect(response).to be_successful
        expect(assigns[:token]).to eq token
        expect(assigns[:token].crypted_token).not_to eq token.crypted_token
        expect(response.body).to match(/#{assigns[:token].full_token}/)
      end

      it "should not allow regenerating a token while masquerading" do
        token = @user.access_tokens.new
        token.developer_key = DeveloperKey.default
        token.save!
        expect(token.user_id).to eq @user.id
        expect(token.protected_token?).to eq false
        Account.site_admin.account_users.create!(user: @user)
        session[:become_user_id] = user_with_pseudonym.id
        put 'update', params: {:id => token.id, :access_token => {:regenerate => '1'}}
        assert_status(401)
      end

      it "should not allow regenerating a protected token" do
        key = DeveloperKey.create!
        token = @user.access_tokens.create!(developer_key: key)
        expect(token.user_id).to eq @user.id
        expect(token.protected_token?).to eq true
        put 'update', params: {:id => token.id, :access_token => {:regenerate => '1'}}
        expect(response).to be_successful
        expect(assigns[:token]).to eq token
        expect(assigns[:token].crypted_token).to eq token.crypted_token
        expect(response.body).not_to match(/#{assigns[:token].token_hint}/)
      end

      it "should not allow updating someone else's token" do
        user2 = User.create!
        token = user2.access_tokens.create!
        expect(token.user_id).to eq user2.id
        put 'update', params: {:id => token.id}
        assert_status(404)
      end
    end
  end
end
