# frozen_string_literal: true

#
# Copyright (C) 2018 - present Instructure, Inc.
#
# This file is part of Canvas.
#
# Canvas is free software: you can redistribute it and/or modify it under
# the terms of the GNU Affero General Public License as published by the Free
# Software Foundation, version 3 of the License.
#
# Canvas is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
# details.
#
# You should have received a copy of the GNU Affero General Public License along
# with this program. If not, see <http://www.gnu.org/licenses/>.

require File.expand_path(File.dirname(__FILE__) + '/../../../spec_helper')
require File.expand_path(File.dirname(__FILE__) + '/concerns/advantage_services_shared_context')
require File.expand_path(File.dirname(__FILE__) + '/concerns/advantage_services_shared_examples')
require File.expand_path(File.dirname(__FILE__) + '/concerns/lti_services_shared_examples')
require_dependency 'lti/ims/scores_controller'

module Lti::Ims
  RSpec.describe ScoresController do
    include_context 'advantage services context'

    let(:context) { course }
    let(:assignment) do
      opts = { course: course }
      if tool.present? && tool.use_1_3?
        opts[:submission_types] = 'external_tool'
        opts[:external_tool_tag_attributes] = {
          url: tool.url, content_type: 'context_external_tool', content_id: tool.id
        }
      end
      assignment_model(opts)
    end
    let(:unknown_context_id) { (Course.maximum(:id) || 0) + 1 }
    let(:line_item) do
      if assignment.external_tool? && tool.use_1_3?
        assignment.line_items.first
      else
        line_item_model(course: course)
      end
    end
    let(:user) { student_in_course(course: course, active_all: true).user }
    let(:line_item_id) { line_item.id }
    let(:result) do
      lti_result_model line_item: line_item, user: user, scoreGiven: nil, scoreMaximum: nil
    end
    let(:submission) { nil }
    let(:json) { JSON.parse(response.body) }
    let(:access_token_scopes) { 'https://purl.imsglobal.org/spec/lti-ags/scope/score' }
    let(:userId) { user.id }
    let(:params_overrides) do
      {
        course_id: context_id,
        line_item_id: line_item_id,
        userId: userId,
        activityProgress: 'Completed',
        gradingProgress: 'FullyGraded',
        timestamp: Time.zone.now.iso8601(3)
      }
    end
    let(:action) { :create }
    let(:scope_to_remove) { 'https://purl.imsglobal.org/spec/lti-ags/scope/score' }

    describe '#create' do
      let(:content_type) { 'application/vnd.ims.lis.v1.score+json' }

      it_behaves_like 'advantage services'
      it_behaves_like 'lti services'

      context 'with valid params' do
        context 'when the lti_id userId is used' do
          let(:userId) { user.lti_id }

          it 'returns a valid resultUrl in the body' do
            send_request
            expect(json['resultUrl']).to include 'results'
          end
        end

        it 'returns a valid resultUrl in the body' do
          send_request
          expect(json['resultUrl']).to include 'results'
        end

        context 'with no existing result' do
          it 'creates a new result' do
            expect { send_request }.to change(Lti::Result, :count).by(1)
          end

          it 'sets the updated_at and created_at to match the params timestamp' do
            send_request
            rslt = Lti::Result.find(json['resultUrl'].split('/').last)
            expect(rslt.created_at).to eq(params_overrides[:timestamp])
            expect(rslt.updated_at).to eq(params_overrides[:timestamp])
          end
        end

        context 'with existing result' do
          context do
            let(:params_overrides) do
              super().merge(scoreGiven: 5.0, scoreMaximum: line_item.score_maximum)
            end

            it 'updates result' do
              result
              expect { send_request }.to change(Lti::Result, :count).by(0)
              expect(result.reload.result_score).to eq 5.0
            end
          end

          it 'sets the updated_at to match the params timestamp' do
            send_request
            rslt = Lti::Result.find(json['resultUrl'].split('/').last)
            expect(rslt.updated_at).to eq(params_overrides[:timestamp])
          end

          context do
            let(:params_overrides) { super().merge(timestamp: 1.day.from_now) }

            it 'does not update the created_at timestamp' do
              result
              send_request
              rslt = Lti::Result.find(json['resultUrl'].split('/').last)
              expect(rslt.created_at).not_to eq(params_overrides[:timestamp])
            end
          end
        end

        context 'when line_item is not an assignment' do
          let(:line_item_no_submission) do
            line_item_model assignment: line_item.assignment,
                            resource_link: line_item.resource_link,
                            tool: tool
          end
          let(:line_item_id) { line_item_no_submission.id }

          context 'with gradingProgress set to FullyGraded or PendingManual' do
            let(:params_overrides) do
              super().merge(scoreGiven: 10, scoreMaximum: line_item.score_maximum)
            end

            it 'does not create submission' do
              send_request
              rslt = Lti::Result.find(json['resultUrl'].split('/').last)
              expect(rslt.submission).to be_nil
            end

            context do
              let(:params_overrides) { super().merge(gradingProgress: 'PendingManual') }

              it 'does not create submission with PendingManual' do
                send_request
                rslt = Lti::Result.find(json['resultUrl'].split('/').last)
                expect(rslt.submission).to be_nil
              end
            end
          end
        end

        context 'when line_item is an assignment' do
          let(:result) { lti_result_model line_item: line_item, user: user }

          shared_examples_for 'creates a new submission' do
            it 'increments attempt' do
              send_request
              attempt = result.submission.reload.attempt
              send_request
              expect(result.submission.reload.attempt).to eq attempt + 1
            end
          end

          shared_examples_for 'updates existing submission' do
            it 'does not increment attempt' do
              send_request
              attempt = result.submission.reload.attempt
              send_request
              expect(result.submission.reload.attempt).to eq attempt
            end
          end

          before { result }

          context 'default behavior' do
            it 'submits homework for module progression' do
              expect_any_instance_of(Assignment).to receive(:submit_homework)
              send_request
            end

            it 'uses submission_type of external_tool' do
              send_request
              expect(result.submission.reload.submission_type).to eq 'external_tool'
            end

            it_behaves_like 'creates a new submission'
          end

          context 'when "new_submission" extension is present and false' do
            let(:params_overrides) do
              super().merge(Lti::Result::AGS_EXT_SUBMISSION => { new_submission: false })
            end

            it 'does not submit homework' do
              expect_any_instance_of(Assignment).to_not receive(:submit_homework)
              expect_any_instance_of(Assignment).to receive(:find_or_create_submission)
              send_request
            end

            it_behaves_like 'updates existing submission'
          end

          context 'when "new_submission" extension is present and true' do
            let(:params_overrides) do
              super().merge(Lti::Result::AGS_EXT_SUBMISSION => { new_submission: true })
            end

            it_behaves_like 'creates a new submission'
          end

          context 'when "submission_type" extension is none' do
            let(:params_overrides) do
              super().merge(Lti::Result::AGS_EXT_SUBMISSION => { submission_type: 'none' })
            end

            it 'does not submit homework' do
              expect_any_instance_of(Assignment).to_not receive(:submit_homework)
              expect_any_instance_of(Assignment).to receive(:find_or_create_submission)
              send_request
            end
          end

          context 'with no scoreGiven' do
            it 'does not update submission' do
              send_request
              expect(result.submission.reload.score).to be_nil
            end
          end

          context 'with gradingProgress not set to FullyGraded or PendingManual' do
            let(:params_overrides) { super().merge(scoreGiven: 100, gradingProgress: 'Pending') }

            it 'does not update submission' do
              send_request
              expect(result.submission.score).to be_nil
            end
          end

          context 'with gradingProgress set to FullyGraded or PendingManual' do
            let(:params_overrides) do
              super().merge(scoreGiven: 10, scoreMaximum: line_item.score_maximum)
            end

            it 'updates submission with FullyGraded' do
              send_request
              expect(result.submission.reload.score).to eq 10.0
            end

            context do
              let(:params_overrides) { super().merge(gradingProgress: 'PendingManual') }

              it 'updates submission with PendingManual' do
                send_request
                expect(result.submission.reload.score).to eq 10.0
              end
            end

            context 'with comment in payload' do
              let(:params_overrides) { super().merge(comment: 'Test coment') }

              it 'creates a new submission_comment' do
                send_request
                expect(result.submission.reload.submission_comments).not_to be_empty
              end
            end

            context 'with submission already graded' do
              let(:result) do
                lti_result_model line_item: line_item,
                                 user: user,
                                 result_score: 100,
                                 result_maximum: 10
              end

              it 'updates submission score' do
                expect(result.submission.score).to eq(100)
                send_request
                expect(result.submission.reload.score).to eq 10.0
              end
            end
          end

          context 'with submitted_at extension' do
            let(:params_overrides) do
              super().merge(Lti::Result::AGS_EXT_SUBMISSION => { submitted_at: submitted_at })
            end

            shared_examples_for 'updates submission time' do
              it do
                send_request
                expect(result.submission.reload.submitted_at).to eq submitted_at
              end
            end

            context 'when submitted_at is prior to submission due date' do
              let(:submitted_at) { 5.minutes.ago.iso8601(3) }

              before { result.submission.update!(cached_due_date: 2.minutes.ago.iso8601(3)) }

              it_behaves_like 'updates submission time'
              it_behaves_like 'creates a new submission'

              it 'does not mark submission late' do
                send_request
                expect(Submission.late.count).to eq 0
              end
            end

            context 'when submitted_at is after submission due date' do
              let(:submitted_at) { 2.minutes.ago.iso8601(3) }

              before { result.submission.update!(cached_due_date: 5.minutes.ago.iso8601(3)) }

              it_behaves_like 'updates submission time'
              it_behaves_like 'creates a new submission'

              it 'marks submission late' do
                send_request
                expect(Submission.late.count).to eq 1
              end
            end

            context 'when new_submission is present and false' do
              let(:submitted_at) { 5.minutes.ago.iso8601(3) }
              let(:params_overrides) do
                super().merge(
                  Lti::Result::AGS_EXT_SUBMISSION => {
                    submitted_at: submitted_at, new_submission: false
                  }
                )
              end

              it_behaves_like 'updates submission time'
              it_behaves_like 'updates existing submission'
            end
          end
        end

        context 'with different scoreMaximum' do
          let(:params_overrides) { super().merge(scoreGiven: 10, scoreMaximum: 100) }

          it 'scales the submission but does not scale the score for the result' do
            result
            send_request
            expect(result.reload.result_score).to eq(params_overrides[:scoreGiven])
            expect(result.submission.reload.score).to eq(
              result.reload.result_score * (line_item.score_maximum / 100)
            )
          end
        end

        context "with a ZERO score maximum" do
          let(:params_overrides) { super().merge(scoreGiven: 0, scoreMaximum: 0) }

          it 'will not tolerate invalid score max' do
            result
            send_request
            expect(response.status.to_i).to eq(422)
          end
        end

        context 'with online_url' do
          let(:params_overrides) do
            super().merge(
              Lti::Result::AGS_EXT_SUBMISSION => {
                submission_type: 'online_url', submission_data: 'http://www.instructure.com'
              }
            )
          end

          it 'updates the submission and result url' do
            result
            send_request
            expect(
              result.reload.extensions[Lti::Result::AGS_EXT_SUBMISSION]['submission_type']
            ).to eq('online_url')
            expect(
              result.reload.extensions[Lti::Result::AGS_EXT_SUBMISSION]['submission_data']
            ).to eq('http://www.instructure.com')
            expect(result.submission.submission_type).to eq('online_url')
            expect(result.submission.url).to eq('http://www.instructure.com')
          end
        end

        context 'with basic_lti_launch' do
          let(:params_overrides) do
            super().merge(
              Lti::Result::AGS_EXT_SUBMISSION => {
                submission_type: 'basic_lti_launch',
                submission_data: 'http://www.instructure.com/launch_url'
              }
            )
          end

          it 'updates the submission and result url' do
            result
            send_request
            expect(
              result.reload.extensions[Lti::Result::AGS_EXT_SUBMISSION]['submission_type']
            ).to eq('basic_lti_launch')
            expect(
              result.reload.extensions[Lti::Result::AGS_EXT_SUBMISSION]['submission_data']
            ).to eq('http://www.instructure.com/launch_url')
            expect(result.submission.submission_type).to eq('basic_lti_launch')
            expect(result.submission.url).to eq('http://www.instructure.com/launch_url')
          end
        end

        context 'with online_text_entry' do
          let(:params_overrides) do
            super().merge(
              Lti::Result::AGS_EXT_SUBMISSION => {
                submission_type: 'online_text_entry', submission_data: '<p>Here is some text</p>'
              }
            )
          end

          it 'updates the submission and result body' do
            result
            send_request
            expect(
              result.reload.extensions[Lti::Result::AGS_EXT_SUBMISSION]['submission_type']
            ).to eq('online_text_entry')
            expect(
              result.reload.extensions[Lti::Result::AGS_EXT_SUBMISSION]['submission_data']
            ).to eq('<p>Here is some text</p>')
            expect(result.submission.submission_type).to eq('online_text_entry')
            expect(result.submission.body).to eq('<p>Here is some text</p>')
          end
        end

        context 'when previously graded and score not given' do
          let(:result) do
            lti_result_model line_item: line_item,
                             user: user,
                             result_score: 100,
                             result_maximum: 200
          end
          let(:params_overrides) { super().except(:scoreGiven, :scoreMaximum) }

          it 'clears the score' do
            expect(result.submission.score).to eq(100)
            expect(result.result_score).to eq(100)
            expect(result.result_maximum).to eq(200)
            send_request
            expect(result.reload.result_score).to be_nil
            expect(result.reload.result_maximum).to be_nil
            expect(result.submission.reload.score).to be_nil
          end
        end
      end

      context 'with invalid params' do
        shared_examples_for 'a bad request' do
          it 'does not process request' do
            result
            send_request
            expect(response).to be_bad_request
          end
        end

        shared_examples_for 'an unprocessable entity' do
          it 'returns an unprocessable_entity error' do
            result
            send_request
            expect(response).to have_http_status :unprocessable_entity
          end
        end

        context 'when timestamp is before updated_at' do
          let(:params_overrides) { super().merge(timestamp: 1.day.ago.iso8601(3)) }
          it_behaves_like 'a bad request'
        end

        context 'when scoreGiven is supplied without scoreMaximum' do
          let(:params_overrides) do
            super().merge(scoreGiven: 10, scoreMaximum: line_item.score_maximum).except(
              :scoreMaximum
            )
          end
          it_behaves_like 'an unprocessable entity'
        end

        context 'when user_id not found in course' do
          let(:user) { student_in_course(course: course_model, active_all: true).user }
          it_behaves_like 'an unprocessable entity'
        end

        context 'when user_id is not a student in course' do
          let(:user) { ta_in_course(course: course, active_all: true).user }
          it_behaves_like 'an unprocessable entity'
        end

        context 'when timestamp is not a string' do
          let(:params_overrides) { super().merge(timestamp: Time.zone.now.to_i) }
          it_behaves_like 'a bad request'
        end

        context 'when submitted_at extension is not a string' do
          let(:params_overrides) do
            super().merge(Lti::Result::AGS_EXT_SUBMISSION => { submitted_at: Time.zone.now.to_i })
          end
          it_behaves_like 'a bad request'
        end

        context 'when submitted_at extension is an invalid timestamp' do
          let(:params_overrides) do
            super().merge(Lti::Result::AGS_EXT_SUBMISSION => { submitted_at: 'asdf' })
          end
          it_behaves_like 'a bad request'
        end

        context 'when submitted_at is in the future' do
          let(:params_overrides) do
            super().merge(
              Lti::Result::AGS_EXT_SUBMISSION => { submitted_at: Time.zone.now + 5.minutes }
            )
          end
          it_behaves_like 'a bad request'
        end
      end
    end
  end
end
