/*
 * Copyright (C) 2019 - present Instructure, Inc.
 *
 * This file is part of Canvas.
 *
 * Canvas is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License as published by the Free
 * Software Foundation, version 3 of the License.
 *
 * Canvas is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License along
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 */

import React from 'react'
import {render, fireEvent} from '@testing-library/react'
import {DateTime} from '@instructure/ui-i18n'
import moment from 'moment'

import EditableDateTime from '../EditableDateTime'

const locale = 'en'
const timeZone = DateTime.browserTimeZone()

function renderEditableDateTime(props = {}) {
  return render(
    <EditableDateTime
      mode="view"
      onChange={() => {}}
      onChangeMode={() => {}}
      value="2019-04-11T13:00:00-05:00"
      label="Due"
      locale={locale}
      timeZone={timeZone}
      readOnly={false}
      placeholder="No due date"
      displayFormat="lll"
      invalidMessage={() => undefined}
      {...props}
    />
  )
}

describe('EditableDateTime', () => {
  it('renders in view mode', () => {
    const value = '2019-04-11T13:00:00-05:00'
    const {getByText} = renderEditableDateTime({value})

    const dtstring = DateTime.toLocaleString(value, locale, timeZone, 'lll')
    expect(getByText('Edit Due')).toBeInTheDocument()
    expect(getByText(dtstring)).toBeInTheDocument()
  })

  it('renders in edit mode', () => {
    const value = '2019-04-11T13:00:00-05:00'
    const {getAllByText, getByLabelText} = renderEditableDateTime({mode: 'edit', value})

    const dtstring = DateTime.toLocaleString(value, locale, timeZone, 'LLL')
    const datestr = DateTime.toLocaleString(value, locale, timeZone, 'LL')
    const timestr = DateTime.toLocaleString(value, locale, timeZone, 'LT')
    expect(getByLabelText('Date').value).toBe(datestr)
    expect(getByLabelText('Time').value).toBe(timestr)
    expect(getAllByText(dtstring)[0]).toBeInTheDocument()
  })

  it('exits edit mode and reverts to previous value on Escape', () => {
    const value = '2019-04-11T13:00:00-05:00'
    const onChangeMode = jest.fn()
    const onChange = jest.fn()
    const {getByDisplayValue} = renderEditableDateTime({
      mode: 'edit',
      onChangeMode,
      onChange,
      value
    })

    const datestr = DateTime.toLocaleString(value, locale, timeZone, 'LL')
    const timestr = DateTime.toLocaleString(value, locale, timeZone, 'LT')
    const dinput = getByDisplayValue(datestr)
    const tinput = getByDisplayValue(timestr)
    // enter a new date
    fireEvent.change(dinput, {target: {value: 'April 4, 2018'}})
    tinput.focus()
    const newDateIsoStr = moment('2018-04-04T13:00:00-05:00').toISOString(true)
    expect(onChange).toHaveBeenLastCalledWith(newDateIsoStr)

    dinput.focus()
    fireEvent.keyUp(dinput, {key: 'Escape', code: 27})
    expect(onChange).toHaveBeenLastCalledWith(value)
    expect(onChangeMode).toHaveBeenLastCalledWith('view')
  })

  // I've spent a day trying to get the folloning specs to work, but I have failed
  // to find the combination of simulated events that get the right event
  // handlers called. Events are getting lost among
  // input -> TextInput -> DateInput -> DateTimeInput -> Editable -> EditableDateTime
  // It works in the UI, but I've failed to simulate the user's input

  // it('saves new value on Enter', async () => {
  //   const value = '2018-04-11T13:00:00-05:00'
  //   const onChange = jest.fn()
  //   const onChangeMode = jest.fn()
  //   const {container} = renderEditableDateTime({
  //     mode: 'edit',
  //     onChange,
  //     onChangeMode,
  //     value
  //   })

  //   const input = container.querySelector('[data-testid="EditableDateTime-editor"] input')
  //   const newdt = toLocaleString('2019-04-11T13:00:00-05:00', locale, timeZone, 'LL')
  //   fireEvent.change(input, {target: {value: newdt}})

  //   fireEvent.keyDown(input, {key: 'Enter', code: 13, target: {value: input.value}})
  //   await wait(() => {
  //     expect(onChangeMode).toHaveBeenCalledWith('view')
  //   })

  //   await wait(() => {
  //     expect(onChange).toHaveBeenCalled()
  //     expect(onChange).toHaveBeenCalledWith('2019-04-11T13:00:00-05:00')
  //   })
  // })

  // it('saves the new value on blur', () => {
  //   const value = '2019-04-11T13:00:00-05:00'
  //   const onChange = jest.fn()
  //   const onChangeMode = jest.fn()
  //   const {container, getByDisplayValue} = render(
  //     <div>
  //       <EditableDateTime
  //         mode="edit"
  //         onChange={onChange}
  //         onChangeMode={onChangeMode}
  //         value={value}
  //         label="Due"
  //         locale={locale}
  //         timeZone={timeZone}
  //         readOnly={false}
  //         placeholder="No due date"
  //         displayFormat="lll"
  //         invalidMessage={() => undefined}
  //       />
  //       <span id="focus-me" tabIndex="-1">
  //         just here to get focus
  //       </span>
  //     </div>
  //   )

  //   let displayValue = toLocaleString(value, locale, timeZone, 'LL')
  //   const input = getByDisplayValue(displayValue)
  //   input.focus()
  //   const newValue = '2019-04-12T13:00:00-05:00'
  //   displayValue = toLocaleString(newValue, locale, timeZone, 'LL')
  //   fireEvent.change(input, {target: {value: displayValue}})
  //   container.querySelector('#focus-me').focus()

  //   expect(onChangeMode).toHaveBeenCalledWith('view')
  //   expect(onChange).toHaveBeenCalledWith(newValue)
  // })

  // it('reverts to the old value on Escape', () => {
  //  // if I can't get change to work, I can't revert...
  // })
})
